#include <map>
#include <string>
#include <vector>

#include "jcfcoreutils/FunctionTask.h"
#include "csfunified/framework/ServicesDispatcher.h"
#include "csfunified/featuresets/IMPresenceFeatureSet.h"
#include "csfunified/featuresets/impsubsets/HttpProxyFeatureSubset.h"
#include "csf/netutils/NetUtilsFactory.hpp"
#include "csf/http/HttpClientListener.hpp"
#include "csf/http/Response.hpp"
#include "csf/http/Request.hpp"
#include "csf/http/ProgressData.hpp"
#include "csf/http/HttpRequestOptions.hpp"
#include "csf/http/EdgePolicy.hpp"
#include "csf/http/Proxy.hpp"
#include "csf/cert/CertVerifierFactory.hpp"
#include "csf/cert/UserInteractingInvalidCertManagementPolicy.hpp"
#include "csf/ScopedLock.hpp"
#include "csf/Mutex.hpp"
#include "csf/Task.hpp"
#include "csf/logger/CSFLogger.hpp"

#include "jcfcoreutils/ThreadUtils.h"

#include "boost/bind.hpp"

#ifdef WIN32
#include "featuresets/Adapters/Dependence.h"
#else
#include "sdkwrapper/mac/MeetingApiDelegate.h"
#endif
#include "NetUtilBridge.h"

using namespace csf::http;
using namespace std;
using namespace CSFUnified;

static CSFLogger* logger = CSFLogger_getLogger("MeetingService-NetUtilBridge");

class JMHttpClientListener: public HttpClientListener
{
public:
	JMHttpClientListener(SMART_PTR_NS::shared_ptr<CSFUnified::UnifiedFactory> unifiedFactory);
	virtual ~JMHttpClientListener();
	void onRequestFailure(unsigned int requestID, HttpClientResult::Result result);
	void onResponseReceived(unsigned int requestID, const Response & response);
	void onProgressUpdated(unsigned int requestID, ProgressData progressData);

	void addRequest(unsigned int requestID, NetUtilSuccessProcT successProc, NetUtilErrorProcT errorProc,
					NetUtilDownloadProcT downloadProc, csf::http::BasicHttpClientPtr httpClient);
    
	void cancelAllRequest();

protected:
	void SuccessProc(unsigned int requestID, Response response);
	void ErrorProc(unsigned int requestID, HttpClientResult::Result result);

private:
	struct JMHttpCallback
	{
		NetUtilSuccessProcT successProc;
		NetUtilErrorProcT errorProc;
		NetUtilDownloadProcT downloadProc;
		csf::http::BasicHttpClientPtr httpClient;
	};
	map<unsigned int, JMHttpCallback> m_mapCB;
	csf::Mutex m_mapMutex;
	CSFUnified::UnifiedFactoryPtr m_unifiedFactory;
}; 

JMHttpClientListener::JMHttpClientListener(SMART_PTR_NS::shared_ptr<CSFUnified::UnifiedFactory> unifiedFactory)
{
	m_unifiedFactory = unifiedFactory;
}

JMHttpClientListener::~JMHttpClientListener()
{
	cancelAllRequest();
}

void JMHttpClientListener::addRequest(unsigned int requestID, NetUtilSuccessProcT successProc, NetUtilErrorProcT errorProc,
									  NetUtilDownloadProcT downloadProc, csf::http::BasicHttpClientPtr httpClient)
{
	if (m_mapCB.find(requestID) != m_mapCB.end())
	{
		CSFLogWarnS(logger, "Found requestID  " << requestID << " already in callback map.");
	}
	JMHttpCallback cb;
	cb.successProc = successProc;
	cb.errorProc = errorProc;
	cb.downloadProc = downloadProc;
	cb.httpClient = httpClient;

	CSFLogInfoS(logger, "Add " << requestID << " to callback map.");

	csf::ScopedLock lock(m_mapMutex);
	m_mapCB[requestID] = cb;
}

void JMHttpClientListener::cancelAllRequest()
{
    vector<unsigned int> allRequestIDs;
    {
        csf::ScopedLock lock(m_mapMutex);
        
        for (map<unsigned int, JMHttpCallback>::iterator iter = m_mapCB.begin(); iter != m_mapCB.end(); iter++)
        {
            allRequestIDs.push_back(iter->first);
        }
    }
    
    for (vector<unsigned int>::size_type i = 0; i < allRequestIDs.size(); i++)
    {
        onRequestFailure(allRequestIDs[i], HttpClientResult::REQUEST_CANCELLED);
    }
}

void JMHttpClientListener::SuccessProc(unsigned int requestID, Response response)
{
	if (m_mapCB.find(requestID) == m_mapCB.end())
	{
		CSFLogWarnS(logger, "Can't find requestID  " << requestID);
		return;
	}

	m_mapCB[requestID].httpClient.reset();

	if (m_mapCB[requestID].successProc != nullptr)
	{
		m_mapCB[requestID].successProc(response.responseCode, response.body);
	}

	if (m_mapCB[requestID].downloadProc != nullptr)
	{
		if (response.responseCode == 200)
		{
			CSFLogDebugS(logger, "Download success");
			m_mapCB[requestID].downloadProc(NetUtilDownloadResultEnum::OK);
		}
		else
		{
			CSFLogWarnS(logger, "Download failed");
			m_mapCB[requestID].downloadProc(NetUtilDownloadResultEnum::DOWNLOAD_FAILED);
		}
	}
	
	csf::ScopedLock lock(m_mapMutex);
	m_mapCB.erase(requestID);
}

void JMHttpClientListener::ErrorProc(unsigned int requestID, HttpClientResult::Result result)
{
	if (m_mapCB.find(requestID) == m_mapCB.end())
	{
		CSFLogWarnS(logger, "Can't find requestID  " << requestID);
		return;
	}

	m_mapCB[requestID].httpClient.reset();

	if (m_mapCB[requestID].errorProc != nullptr)
	{
		m_mapCB[requestID].errorProc((int)result);
	}

	if (m_mapCB[requestID].downloadProc != nullptr)
	{
		m_mapCB[requestID].downloadProc(NetUtilDownloadResultEnum::DOWNLOAD_FAILED);
	}

	csf::ScopedLock lock(m_mapMutex);
	m_mapCB.erase(requestID);
}

void JMHttpClientListener::onRequestFailure(unsigned int requestID, HttpClientResult::Result result)
{
	CSFLogWarnS(logger, "Request " << requestID << " fail " << result);
	if (NULL == m_unifiedFactory)
	{
		CSFLogWarnS(logger, "m_unifiedFactory is NULL");
		return;
	}

	if(m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess())
	{
		ErrorProc(requestID, result);
	}
	else
	{
		csf::TaskPtr task(new JCFCoreUtils::FunctionTask(
		   std::bind(&JMHttpClientListener::ErrorProc, this, requestID, result), "JMHttpClientListener::ErrorProc"));
		m_unifiedFactory->getServicesDispatcher()->enqueueBlock(task);
	}
}

void JMHttpClientListener::onResponseReceived(unsigned int requestID, const Response & response)
{
	CSFLogDebugS(logger, "Request success " << response.responseCode);
	if (NULL == m_unifiedFactory)
	{
		CSFLogWarnS(logger, "m_unifiedFactory is NULL");
		return;
	}

	if(m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess())
	{
		SuccessProc(requestID, response);
	}
	else
	{
		csf::TaskPtr task(new JCFCoreUtils::FunctionTask(
		   std::bind(&JMHttpClientListener::SuccessProc, this, requestID, response), "JMHttpClientListener::SuccessProc"));
		m_unifiedFactory->getServicesDispatcher()->enqueueBlock(task);
	}
}

void JMHttpClientListener::onProgressUpdated(unsigned int requestID, ProgressData response)
{
	//Nothing to do
}

NetUtilTransport::NetUtilTransport(SMART_PTR_NS::shared_ptr<CSFUnified::UnifiedFactory> unifiedFactory)
{
	m_pListener.reset(new JMHttpClientListener(unifiedFactory));
	m_unifiedFactory = unifiedFactory;
	m_bCupMode = false;
#ifdef WIN32
	if (CDependence::getInstance()->GetJabberWerxCommonHelper()->IsCupMode())
#else
	if (CSFUnified::MeetingApiDelegate::getInstance()->isCupModel())
#endif
	{
		m_bCupMode = true;
	}
}

NetUtilTransport::~NetUtilTransport()
{
}


void NetUtilTransport::CancelAllRequest()
{
	CSFLogDebugS(logger, "Cancel all request");
	m_pListener->cancelAllRequest();
}

void NetUtilTransport::BuildRequest(RequestPtr request, const vector<string>& httpHeader, const string& httpPostData, JMHttpMethod httpMethod)
{
	request->useSystemProxy(true);
	//Need to fill proxy authentication information
	ProxyPtr proxy = request->getProxy();
	if (m_unifiedFactory != NULL && proxy != NULL)
	{
		SMART_PTR_NS::shared_ptr<IMPresenceFeatureSet> imPresenceFeatureSet = m_unifiedFactory->getFeatureSet<IMPresenceFeatureSet>();
		if (imPresenceFeatureSet && imPresenceFeatureSet->getHttpProxySubset())
		{
			if (imPresenceFeatureSet->getHttpProxySubset()->isCredentialCorrect())
			{
				std::string username;
				csf::SecureString password;
				imPresenceFeatureSet->getHttpProxySubset()->GetProxyCredential(username, password);
				proxy->setAuthentication(username, password, ProxyAuthentication::ANY);
				request->setProxy(proxy);
			}
		}
	}

	request->getOptions()->body = httpPostData; 
	request->getOptions()->connectionTimeout = 30 * 1000;  //30s
	request->getOptions()->transferTimeout = 30 * 1000; //30s
	request->getOptions()->forbidConnectionReuse = true;

	if (httpMethod == JM_HTTP_POST)
	{
		request->getOptions()->method = HttpMethod::HTTP_POST;
	}
	else
	{
		request->getOptions()->method = HttpMethod::HTTP_GET;
	}

	HeaderVecPtr headers(new vector<HeaderType>());
	for (size_t i = 0; i < httpHeader.size(); i++)
	{
		size_t pos = httpHeader[i].find(":");
		if (pos != string::npos)
		{
			headers->push_back(HeaderType(httpHeader[i].substr(0, pos), httpHeader[i].substr(pos + 1)));
		}
	}
	request->getOptions()->headerList = headers;
}

csf::http::BasicHttpClientPtr NetUtilTransport::GetHttpClient()
{
	csf::http::BasicHttpClientPtr pHttpClient;
	if (m_bCupMode)
	{
		pHttpClient = csf::netutils::NetUtilsFactory().createBasicHttpClient();
	}
	else
	{
		csf::cert::CertVerifierPtr certVerifier = csf::cert::CertVerifierFactory::createCertVerifier();
		csf::cert::UserInteractingInvalidCertManagementPolicy::NotifyingFailingAvoidableUserInteractingInvalidCertManagementPolicyBuilder builder;
		certVerifier->addPolicy(builder.build());
		pHttpClient = csf::netutils::NetUtilsFactory().createBasicHttpClient(certVerifier);
	}
	pHttpClient->addPolicy(csf::http::EdgePolicy::NEVER_USE_EDGE_POLICY);
	return pHttpClient;
}

void NetUtilTransport::HttpRequest(const string& httpUrl, const vector<string>& httpHeader, const string& httpPostData, JMHttpMethod httpMethod, NetUtilSuccessProcT successProc, NetUtilErrorProcT errorProc)
{
	CSFLogDebugS(logger, "Request to " << httpUrl);

	csf::http::BasicHttpClientPtr pHttpClient = GetHttpClient();

	RequestPtr request(new Request(httpUrl));
	BuildRequest(request, httpHeader, httpPostData, httpMethod);

	Response response;
	HttpClientResult::Result result = pHttpClient->execute(request, response); 

	if (result == 0 && successProc != nullptr)
	{
		successProc(response.responseCode, response.body);
	}
	else if (result != 0 && errorProc != nullptr)
	{
		errorProc((int)result);
	}
}

void NetUtilTransport::HttpRequestAsync(const string& httpUrl, const vector<string>& httpHeader, const string& httpPostData, JMHttpMethod httpMethod, NetUtilSuccessProcT successProc, NetUtilErrorProcT errorProc)
{
	CSFLogDebugS(logger, "Request to " << httpUrl);

	csf::http::BasicHttpClientPtr pHttpClient = GetHttpClient();

	RequestPtr request(new Request(httpUrl));
	BuildRequest(request, httpHeader, httpPostData, httpMethod);

	request->setListener(m_pListener);

	unsigned int requestID;
	HttpClientError::Error result = pHttpClient->executeAsync(request, requestID); 
	if (result == 0)
	{
		m_pListener->addRequest(requestID, successProc, errorProc, nullptr, pHttpClient);
	}
	else
	{
		CSFLogWarnS(logger, "executeAsync fail " << result);
	}
}

NetUtilTransportDownloader::NetUtilTransportDownloader(SMART_PTR_NS::shared_ptr<CSFUnified::UnifiedFactory> unifiedFactory) : NetUtilTransport(unifiedFactory)
{
    // noting to do
}

NetUtilTransportDownloader::~NetUtilTransportDownloader()
{
}

void NetUtilTransportDownloader::DownloadToFile(std::string url, std::string filePath, NetUtilDownloadProcT downloadProc, long timeout)
{
	if(url.empty())
	{
		CSFLogErrorS(logger, "Empty url, unable to request download");
		downloadProc(NetUtilDownloadResultEnum::PARAMETER_ERROR);
		return;
	}

	if(filePath.empty())
	{
		CSFLogErrorS(logger, "Empty file path, unable to create file for download: " <<  filePath);
		downloadProc(NetUtilDownloadResultEnum::FILE_ERROR);
		return;
	}

	// Execute download on new thread
	JCFCoreUtils::ThreadUtils::runInThreadPool(boost::bind(&NetUtilTransportDownloader::backgroundDownloadToFile, this, url, filePath, downloadProc), "NetUtilTransport::DownloadToFile");
}

void NetUtilTransportDownloader::backgroundDownloadToFile(std::string url, const std::string& filePath, NetUtilDownloadProcT downloadProc)
{
	CSFLogDebugS(logger, "Entering backgroundDownloadToFile(url[" << url << "], file[" << filePath << "])");

	csf::http::BasicHttpClientPtr pHttpClient = GetHttpClient();

	RequestPtr request(new Request(url));
	std::vector<std::string> header;
	BuildRequest(request, header, "", JM_HTTP_GET);
	request->getOptions()->transferTimeout = 10 * 60 * 1000; //10min
	request->setListener(m_pListener);

	unsigned int requestID;
	HttpClientError::Error result = pHttpClient->downloadAsync(request, requestID, filePath);
	if (result == 0)
	{
		m_pListener->addRequest(requestID, nullptr, nullptr, downloadProc, pHttpClient);
	}
	else
	{
		CSFLogWarnS(logger, "downloadAsync fail " << result);
		downloadProc(NetUtilDownloadResultEnum::DOWNLOAD_FAILED);
	}

	CSFLogDebugS(logger, "Exiting backgroundDownloadToFile()");
}
